# AmirHosein Sadeghimanesh
# 2020 February, modified 2021 August
#
# This Julia file contains the computation for applying Algorithm 1 of the paper on the example in
# Section 5.2. We get help from the Kac-Rice integrals and the codes in the repository of
# the paper [1]. The link to the repository: [2]. (See the references at the end of this file)
#
# Note: To compile this code, Julia's package "Distributions" should be installed.
# 
# First we define a Julia function corresponding to the Algorithm 1.
# 
# We need an extra function before defining the main function.
# 
# The SameBox function returns a copy of the input arbitrary-type array without
# making them dependent on each other.
# 
function SameBox(B)
    return(deepcopy(B))
end
# 
# 
# Note: Insert the coordinates of the initial box in 'Float' format. Otherwise Julia fixes their types to
# 'Int', and when bisecting results to a 'Float' number, Julia stops the algorithm and rises an error 
# like "InexactError: Int64(2.5)" etc.
# 
# Therefore even if an input number is an integer, just put a decimal point and a zero in front
# of it. Please do not choose a very small value for terminate_condition if you do not want to wait
# for a long time.
# 
@fastmath function BisectDescribtion(B, g, n1, n2, terminate_condition)
    st1 = time_ns() # The start time of the computation in nano second unit.
    r = length(B)  # B is considered as list of intervals which their Cartesian product is B. Therefore length of B or equivalenty the number of intervals is the number of variables (parameters in CRN case).
    List1 = [] # List of subhyperrectangles K that g(K)=n1.
    List2 = [] # List of subhyperrectangles K that g(K)=n2.
    S = [[B,1]] # List of undecided subhyperrectangles. 
    n1_threshold = n1 + (n2 - n1) / 20 # Since later we want to use a function g which is a random and an approximate, we replace the condition g(K)=n1 with g(K) close to n1. The user can change this condition or choose a different n1_threshold.
    n2_threshold = n2 - (n2 - n1) / 20 # The threshold for deciding g(K) is approximately equal to n2.
    number_of_evaluations = 0 # To keep track of how many evaluatiosn of g(K) haa happened until the algorithm terminates completely.
    @inbounds while length(S) != 0
        K_and_i = S[1]
        deleteat!(S, 1)
        K = K_and_i[1]
        axis_index = K_and_i[2]
        number_of_evaluations += 1
        ans0 = g(K)
        if ans0 < n1_threshold
            push!(List1, SameBox(K))
        elseif ans0 > n2_threshold
            push!(List2, SameBox(K))
        else
            # edge_index = 1
            # terminate_condition_happened = 0
            # while terminate_condition_happened == 0 && edge_index < r + 1
            #     if K[edge_index][2] - K[edge_index][1] < terminate_condition
            #         terminate_condition_happened = 1
            #     end
            #     edge_index += 1
            # end
            terminate_condition_happened = 0
            if ((K[axis_index][2] - K[axis_index][1]) / 2) < terminate_condition
                terminate_condition_happened = 1
            end
            if terminate_condition_happened == 1
                if ans0 > (n2 + n1) / 2
                    push!(List2, SameBox(K))
                else
                    push!(List1, SameBox(K))
                end
            else
                K1 = SameBox(K)
                K1[axis_index][2] = (K[axis_index][1] + K[axis_index][2]) / 2
                K2 = SameBox(K)
                K2[axis_index][1] = (K[axis_index][1] + K[axis_index][2]) / 2
                if axis_index == r
                    axis_index = 1
                else
                    axis_index += 1
                end
                push!(S, [K1,axis_index])
                push!(S, [K2,axis_index])
            end
        end
    end
    st2 = time_ns() # The end time of the computation in nano second.
    return List1, List2, (st2 - st1) / 10^9, number_of_evaluations, S  # By dividing tp 10^9, we get the time in second unit instead of nano second unit.
end
# 
# Now defining the function g which gives the average number of steady states for the network in the example.
# 
# We use the Julia package 'Distribution' to generate uniform sample from an interval.
# 
import Distributions: Uniform
# 
function samplor(a, b) # Generating a sample from uniform distribution on an interval.
   	return rand(Uniform(a, b))
end
# 
function chi(a, b, x) # Indicator function on an interval.
   	if x < a
      		return 0
   	elseif x > b
      		return 0
   	else
      		return 1
   	end
end
# 
# Read the reference [1] to know what are the following functions. You can also check the codes [2] of the paper [1].
# 
@fastmath function h1(t)
   	return 366450 * t^2 + 537142.41 * t
end
@fastmath function beta1(t)
   	return 2518322.5 * t^2 + 63502.1205 * t + 26857.1205
end
@fastmath function g1(t, T2)
   	return beta1(t) * (T2 - t) / h1(t)
end
@fastmath function detJf(t, T1, T2)
   	return 537142.41 * T1 - 63502.1205 * T2 + 7554967.5 * t^2 + 2 * t * (366450 * T1 - 2518322.5 * T2 + 63502.1205) + 26857.1205
end
@fastmath function J(t, T2)
   	return abs(detJf(t, g1(t, T2), T2))
end
@fastmath function IntegrandWithoutCOEFF(t, aT1, bT1, T2)
   	return J(t, T2) * chi(aT1, bT1, g1(t, T2)) / h1(t)
end
@fastmath function sumo_antithetic_fun(B)
   	NN = 10^7
   	TT1 = B[1]
   	TT2 = B[2]
   	Ians = 0
   	centero = [(0 + TT2[2]) / 2,(TT2[1] + TT2[2]) / 2]
   	COEFF = TT2[2] / (TT1[2] - TT1[1])
   	t = samplor(0, TT2[2])
   	T2 = samplor(TT2[1], TT2[2])
   	Ians += IntegrandWithoutCOEFF(t, TT1[1], TT1[2], T2)
   	Ians = COEFF * Ians
   	t = 2 * centero[1] - t
   	T2 = 2 * centero[2] - T2
   	delto = COEFF * IntegrandWithoutCOEFF(t, TT1[1], TT1[2], T2) - Ians
   	Ians += delto / 2
   	@inbounds for i = 1:floor(NN / 2)
      		t = samplor(0, TT2[2])
      		T2 = samplor(TT2[1], TT2[2])
      		delto = COEFF * IntegrandWithoutCOEFF(t, TT1[1], TT1[2], T2) - Ians
      		Ians += delto / (2 * i + 1)
      		t = 2 * centero[1] - t
      		T2 = 2 * centero[2] - T2
      		delto = COEFF * IntegrandWithoutCOEFF(t, TT1[1], TT1[2], T2) - Ians
      		Ians += delto / (2 * i + 2)
   	end
   	return Ians
end
# 
# Toggle to comment all Examples below except one of them at a time.
# 
# Example number 1.
#
# Now finding the two-valued bisect describtion of the multistationarity region on the hyperrectangle [1,3]x[2,4] with the terminate_condition being 1. Since the edge length of subhyperrectangles in this method are equal to length of edges of B divided by a power of 2, choosing terminate_condition=1, will tells us that minimum of length of egdes of K_i's in this representation will be at least 1.
# 
B_example_1 = [[1.0,3.0],[2.0,4.0]]
terminate_condition_example_1 = 1.0
n1_example_1 = 1.0
n2_example_1 = 3.0
ANS = BisectDescribtion(B_example_1, sumo_antithetic_fun, n1_example_1, n2_example_1, terminate_condition_example_1)
# 
# Now writing the result in a txt file. 
#
Lines_list = "The output of the two-valued bisect representation\n\n"
Lines_list = Lines_list * "B: [" * string(B_example_1[1][1]) * "," * string(B_example_1[2][1]) * "," * string(B_example_1[1][2]) * "," * string(B_example_1[2][2]) * "]\n\n"
Lines_list = Lines_list * "terminate condition: " * string(terminate_condition_example_1) * "\n"
Lines_list = Lines_list * "n1: " * string(n1_example_1) * "\n"
Lines_list = Lines_list * "n2: " * string(n2_example_1) * "\n\n"
Lines_list = Lines_list * "Computation time: " * string(ANS[3]) * "\n"
Lines_list = Lines_list * "Number of integrals: " * string(ANS[4]) * "\n\n"
Lines_list = Lines_list * "K: " * string(length(ANS[2])) * "\n"
for i = 1:length(ANS[2])
    global Lines_list = Lines_list * "[" * string(ANS[2][i][1][1]) * "," * string(ANS[2][i][2][1]) * "," * string(ANS[2][i][1][2]) * "," * string(ANS[2][i][2][2]) * "]\n" # Note that Julia can't change a global variable that was defined outside of the for loop inside of a for loop which is not inside of a function. Therefore when using a for-loop outside of a function and wanting to change something which was defined outside of it, you should add a "global" at the beginning of that line. Here without "global", Julia will not change Lines_list in this for-loop and instead gives an error saying Lines_list is not defined.
end
Lines_list = Lines_list * "K end\n\n"
Lines_list = Lines_list * "List 1: " * string(length(ANS[1])) * "\n"
print(Lines_list)
for i = 1:length(ANS[1])
    print(Lines_list)
    global Lines_list = Lines_list * "[" * string(ANS[1][i][1][1]) * "," * string(ANS[1][i][2][1]) * "," * string(ANS[1][i][1][2]) * "," * string(ANS[1][i][2][2]) * "]\n"
end
Lines_list = Lines_list * "List 1 end\n\n"
Lines_list = Lines_list * "Undecided list (S): " * string(length(ANS[5])) * "\n"
for i = 1:length(ANS[5])
    global Lines_list = Lines_list * "[" * string(ANS[5][i][1][1]) * "," * string(ANS[5][i][2][1]) * "," * string(ANS[5][i][1][2]) * "," * string(ANS[5][i][2][2]) * "]\n"
end
Lines_list = Lines_list * "List 1 end\n\n"
OutputFile = open(joinpath(@__DIR__, "Two_valued_Bisect_Representation_HK_1v2p_n1.txt"), "w")
write(OutputFile, Lines_list)
close(OutputFile)
#
# Example number 2.
# 
# Now finding the two-valued bisect describtion of the multistationarity region on the hyperrectangle [1,3]x[2,4] with the terminate_condition being 0.5. Since the edge length of subhyperrectangles in this method are equal to length of edges of B divided by a power of 2, choosing terminate_condition=1, will tells us that minimum of length of egdes of K_i's in this representation will be at least 0.5.
# 
B_example_1 = [[1.0,3.0],[2.0,4.0]]
terminate_condition_example_1 = 0.5
n1_example_1 = 1.0
n2_example_1 = 3.0
ANS = BisectDescribtion(B_example_1, sumo_antithetic_fun, n1_example_1, n2_example_1, terminate_condition_example_1)
# 
# Now writing the result in a txt file.
# 
Lines_list = "The output of the two-valued bisect representation\n\n"
Lines_list = Lines_list * "B: [" * string(B_example_1[1][1]) * "," * string(B_example_1[2][1]) * "," * string(B_example_1[1][2]) * "," * string(B_example_1[2][2]) * "]\n\n"
Lines_list = Lines_list * "terminate condition: " * string(terminate_condition_example_1) * "\n"
Lines_list = Lines_list * "n1: " * string(n1_example_1) * "\n"
Lines_list = Lines_list * "n2: " * string(n2_example_1) * "\n\n"
Lines_list = Lines_list * "Computation time: " * string(ANS[3]) * "\n"
Lines_list = Lines_list * "Number of integrals: " * string(ANS[4]) * "\n\n"
Lines_list = Lines_list * "K: " * string(length(ANS[2])) * "\n"
for i = 1:length(ANS[2])
    global Lines_list = Lines_list * "[" * string(ANS[2][i][1][1]) * "," * string(ANS[2][i][2][1]) * "," * string(ANS[2][i][1][2]) * "," * string(ANS[2][i][2][2]) * "]\n"
end
Lines_list = Lines_list * "K end\n\n"
Lines_list = Lines_list * "List 1: " * string(length(ANS[1])) * "\n"
print(Lines_list)
for i = 1:length(ANS[1])
    print(Lines_list)
    global Lines_list = Lines_list * "[" * string(ANS[1][i][1][1]) * "," * string(ANS[1][i][2][1]) * "," * string(ANS[1][i][1][2]) * "," * string(ANS[1][i][2][2]) * "]\n"
end
Lines_list = Lines_list * "List 1 end\n\n"
Lines_list = Lines_list * "Undecided list (S): " * string(length(ANS[5])) * "\n"
for i = 1:length(ANS[5])
    global Lines_list = Lines_list * "[" * string(ANS[5][i][1][1]) * "," * string(ANS[5][i][2][1]) * "," * string(ANS[5][i][1][2]) * "," * string(ANS[5][i][2][2]) * "]\n"
end
Lines_list = Lines_list * "List 1 end\n\n"
OutputFile = open(joinpath(@__DIR__, "Two_valued_Bisect_Representation_HK_1v2p_n2.txt"), "w")
write(OutputFile, Lines_list)
close(OutputFile)
#
# Example number 3.
# 
# Now finding the two-valued bisect describtion of the multistationarity region on the hyperrectangle [0,5]x[0,5] with the terminate_condition being 2. Since the edge length of subhyperrectangles in this method are equal to length of edges of B divided by a power of 2, choosing terminate_condition=2, will tells us that minimum of length of egdes of K_i's in this representation will be at least 2.5.
# 
B_example_1 = [[0.0,5.0],[0.0,5.0]]
terminate_condition_example_1 = 2.0
n1_example_1 = 1.0
n2_example_1 = 3.0
ANS = BisectDescribtion(B_example_1, sumo_antithetic_fun, n1_example_1, n2_example_1, terminate_condition_example_1)
# 
# Now writing the result in a txt file.
# 
Lines_list = "The output of the two-valued bisect representation\n\n"
Lines_list = Lines_list * "B: [" * string(B_example_1[1][1]) * "," * string(B_example_1[2][1]) * "," * string(B_example_1[1][2]) * "," * string(B_example_1[2][2]) * "]\n\n"
Lines_list = Lines_list * "terminate condition: " * string(terminate_condition_example_1) * "\n"
Lines_list = Lines_list * "n1: " * string(n1_example_1) * "\n"
Lines_list = Lines_list * "n2: " * string(n2_example_1) * "\n\n"
Lines_list = Lines_list * "Computation time: " * string(ANS[3]) * "\n"
Lines_list = Lines_list * "Number of integrals: " * string(ANS[4]) * "\n\n"
Lines_list = Lines_list * "K: " * string(length(ANS[2])) * "\n"
for i = 1:length(ANS[2])
    global Lines_list = Lines_list * "[" * string(ANS[2][i][1][1]) * "," * string(ANS[2][i][2][1]) * "," * string(ANS[2][i][1][2]) * "," * string(ANS[2][i][2][2]) * "]\n"
end
Lines_list = Lines_list * "K end\n\n"
Lines_list = Lines_list * "List 1: " * string(length(ANS[1])) * "\n"
print(Lines_list)
for i = 1:length(ANS[1])
    print(Lines_list)
    global Lines_list = Lines_list * "[" * string(ANS[1][i][1][1]) * "," * string(ANS[1][i][2][1]) * "," * string(ANS[1][i][1][2]) * "," * string(ANS[1][i][2][2]) * "]\n"
end
Lines_list = Lines_list * "List 1 end\n\n"
Lines_list = Lines_list * "Undecided list (S): " * string(length(ANS[5])) * "\n"
for i = 1:length(ANS[5])
    global Lines_list = Lines_list * "[" * string(ANS[5][i][1][1]) * "," * string(ANS[5][i][2][1]) * "," * string(ANS[5][i][1][2]) * "," * string(ANS[5][i][2][2]) * "]\n"
end
Lines_list = Lines_list * "List 1 end\n\n"
OutputFile = open(joinpath(@__DIR__, "Two_valued_Bisect_Representation_HK_1v2p_n3.txt"), "w")
write(OutputFile, Lines_list)
close(OutputFile)
#
# Example number 4.
# 
# Now finding the two-valued bisect describtion of the multistationarity region on the hyperrectangle [0,5]x[0,5] with the terminate_condition being 1. Since the edge length of subhyperrectangles in this method are equal to length of edges of B divided by a power of 2, choosing terminate_condition=1, will tells us that minimum of length of egdes of K_i's in this representation will be at least 1.25.
# 
B_example_1 = [[0.0,5.0],[0.0,5.0]]
terminate_condition_example_1 = 1.0
n1_example_1 = 1.0
n2_example_1 = 3.0
ANS = BisectDescribtion(B_example_1, sumo_antithetic_fun, n1_example_1, n2_example_1, terminate_condition_example_1)
# 
# Now writing the result in a txt file.
# 
Lines_list = "The output of the two-valued bisect representation\n\n"
Lines_list = Lines_list * "B: [" * string(B_example_1[1][1]) * "," * string(B_example_1[2][1]) * "," * string(B_example_1[1][2]) * "," * string(B_example_1[2][2]) * "]\n\n"
Lines_list = Lines_list * "terminate condition: " * string(terminate_condition_example_1) * "\n"
Lines_list = Lines_list * "n1: " * string(n1_example_1) * "\n"
Lines_list = Lines_list * "n2: " * string(n2_example_1) * "\n\n"
Lines_list = Lines_list * "Computation time: " * string(ANS[3]) * "\n"
Lines_list = Lines_list * "Number of integrals: " * string(ANS[4]) * "\n\n"
Lines_list = Lines_list * "K: " * string(length(ANS[2])) * "\n"
for i = 1:length(ANS[2])
    global Lines_list = Lines_list * "[" * string(ANS[2][i][1][1]) * "," * string(ANS[2][i][2][1]) * "," * string(ANS[2][i][1][2]) * "," * string(ANS[2][i][2][2]) * "]\n"
end
Lines_list = Lines_list * "K end\n\n"
Lines_list = Lines_list * "List 1: " * string(length(ANS[1])) * "\n"
print(Lines_list)
for i = 1:length(ANS[1])
    print(Lines_list)
    global Lines_list = Lines_list * "[" * string(ANS[1][i][1][1]) * "," * string(ANS[1][i][2][1]) * "," * string(ANS[1][i][1][2]) * "," * string(ANS[1][i][2][2]) * "]\n"
end
Lines_list = Lines_list * "List 1 end\n\n"
Lines_list = Lines_list * "Undecided list (S): " * string(length(ANS[5])) * "\n"
for i = 1:length(ANS[5])
    global Lines_list = Lines_list * "[" * string(ANS[5][i][1][1]) * "," * string(ANS[5][i][2][1]) * "," * string(ANS[5][i][1][2]) * "," * string(ANS[5][i][2][2]) * "]\n"
end
Lines_list = Lines_list * "List 1 end\n\n"
OutputFile = open(joinpath(@__DIR__, "Two_valued_Bisect_Representation_HK_1v2p_n4.txt"), "w")
write(OutputFile, Lines_list)
close(OutputFile)
# 
# Example number 5.
# 
# Now finding the two-valued bisect describtion of the multistationarity region on the hyperrectangle [0,5]x[0,5] with the terminate_condition being 1. Since the edge length of subhyperrectangles in this method are equal to length of edges of B divided by a power of 2, choosing terminate_condition=1, will tells us that minimum of length of egdes of K_i's in this representation will be at least 1.25.
# 
B_example_1 = [[0.0,5.0],[0.0,5.0]]
terminate_condition_example_1 = 0.5
n1_example_1 = 1.0
n2_example_1 = 3.0
ANS = BisectDescribtion(B_example_1, sumo_antithetic_fun, n1_example_1, n2_example_1, terminate_condition_example_1)
# 
# Now writing the result in a txt file.
# 
Lines_list = "The output of the two-valued bisect representation\n\n"
Lines_list = Lines_list * "B: [" * string(B_example_1[1][1]) * "," * string(B_example_1[2][1]) * "," * string(B_example_1[1][2]) * "," * string(B_example_1[2][2]) * "]\n\n"
Lines_list = Lines_list * "terminate condition: " * string(terminate_condition_example_1) * "\n"
Lines_list = Lines_list * "n1: " * string(n1_example_1) * "\n"
Lines_list = Lines_list * "n2: " * string(n2_example_1) * "\n\n"
Lines_list = Lines_list * "Computation time: " * string(ANS[3]) * "\n"
Lines_list = Lines_list * "Number of integrals: " * string(ANS[4]) * "\n\n"
Lines_list = Lines_list * "K: " * string(length(ANS[2])) * "\n"
for i = 1:length(ANS[2])
    global Lines_list = Lines_list * "[" * string(ANS[2][i][1][1]) * "," * string(ANS[2][i][2][1]) * "," * string(ANS[2][i][1][2]) * "," * string(ANS[2][i][2][2]) * "]\n"
end
Lines_list = Lines_list * "K end\n\n"
Lines_list = Lines_list * "List 1: " * string(length(ANS[1])) * "\n"
print(Lines_list)
for i = 1:length(ANS[1])
    print(Lines_list)
    global Lines_list = Lines_list * "[" * string(ANS[1][i][1][1]) * "," * string(ANS[1][i][2][1]) * "," * string(ANS[1][i][1][2]) * "," * string(ANS[1][i][2][2]) * "]\n"
end
Lines_list = Lines_list * "List 1 end\n\n"
Lines_list = Lines_list * "Undecided list (S): " * string(length(ANS[5])) * "\n"
for i = 1:length(ANS[5])
    global Lines_list = Lines_list * "[" * string(ANS[5][i][1][1]) * "," * string(ANS[5][i][2][1]) * "," * string(ANS[5][i][1][2]) * "," * string(ANS[5][i][2][2]) * "]\n"
end
Lines_list = Lines_list * "List 1 end\n\n"
OutputFile = open(joinpath(@__DIR__, "Two_valued_Bisect_Representation_HK_1v2p_n5.txt"), "w")
write(OutputFile, Lines_list)
close(OutputFile)
# 
# References
#
# [1]: Elisenda Feliu and AmirHosein Sadeghimanesh. Kac-Rice formulas and the number of solutions of 
# parametrized systems of polynomial equations. arXiv, 2020, https://arxiv.org/abs/2010.00804.
# [2]: AmirHosein Sadeghimanesh and Elisenda Feliu. MCKR project, version 1.0.0. Available online at 
# https://doi.org/10.5281/zenodo.4026954, 2020.
#
# End of the file.